<?php
// bank-transfer.php — دفع عبر التحويل البنكي (محسّنة + آمنة + ديناميكية)
if (session_status() === PHP_SESSION_NONE) session_start();

// التحقق من تسجيل الدخول
if (!isset($_SESSION['user_id'])) {
    $_SESSION['redirect_after_login'] = "bank-transfer.php";
    header("Location: login.php?msg=login_required");
    exit;
}

require_once __DIR__ . "/config/db.php";
require_once __DIR__ . "/config/price.php";

// CSRF بسيط
if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));

$pdo    = db();
$prices = getPrices();

// قراءة السلة (ندعم شكلين: [id=>qty] أو [id=>['qty'=>..,'price'=>..]])
$cartRaw = $_SESSION['cart'] ?? [];
if (!$cartRaw || !is_array($cartRaw)) {
    include __DIR__ . "/includes/header.php";
    echo "<section class='pay-wrap' dir='rtl'><h2 class='section-title'>💳 الدفع عبر التحويل البنكي</h2><div class='note-empty'>🛒 لا توجد منتجات في السلة</div></section>";
    include __DIR__ . "/includes/footer.php";
    exit;
}

// توحيد السلة: [id => qty] + استخراج قائمة المعرفات
$cart = [];
foreach ($cartRaw as $k => $v) {
    if (is_array($v)) {
        $qty = (int)($v['qty'] ?? 0);
    } else {
        $qty = (int)$v;
    }
    if ($qty > 0) $cart[(int)$k] = $qty;
}
if (!$cart) {
    include __DIR__ . "/includes/header.php";
    echo "<section class='pay-wrap' dir='rtl'><h2 class='section-title'>💳 الدفع عبر التحويل البنكي</h2><div class='note-empty'>🛒 لا توجد منتجات في السلة</div></section>";
    include __DIR__ . "/includes/footer.php";
    exit;
}

$ids = array_keys($cart);
$placeholders = implode(',', array_fill(0, count($ids), '?'));

// جلب تفاصيل المنتجات
$stmt = $pdo->prepare("SELECT id,name,image,weight,karat,metal FROM products WHERE id IN ($placeholders)");
$stmt->execute($ids);
$dbrows = $stmt->fetchAll(PDO::FETCH_ASSOC);

// مساعد الصورة
function img_or_placeholder($path) {
    if (!$path) return "assets/img/placeholder-hero.jpg";
    $rel = (strpos($path,'uploads/')===0) ? $path : "uploads/products/".ltrim($path,'/');
    return is_file(__DIR__ . "/".$rel) ? $rel : "assets/img/placeholder-hero.jpg";
}
function e($v){ return htmlspecialchars((string)$v, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }

// بناء عناصر السلة للعرض/الحساب
$items = [];
$total = 0.0;

foreach ($dbrows as $p) {
    $pid   = (int)$p['id'];
    $qty   = (int)($cart[$pid] ?? 0);
    if ($qty <= 0) continue;

    $isGold= ($p['metal'] ?? 'gold') === 'gold';
    $karat = (int)($p['karat'] ?? 24);
    $wt    = (float)($p['weight'] ?? 0.0);

    // السعر/جم بحسب النوع
    $key   = $isGold ? ("gold_".$karat) : "silver";
    $unit  = (isset($prices[$key]) && is_numeric($prices[$key])) ? (float)$prices[$key] : 0.0;

    // دعم الشكل القديم: إن وُجد سعر جاهز في السلة نستخدمه كسعر القطعة
    $legacyPiece = null;
    if (is_array($cartRaw[$pid] ?? null)) {
        $legacyPiece = isset($cartRaw[$pid]['price']) ? (float)$cartRaw[$pid]['price'] : null;
    }

    $piece = ($legacyPiece !== null) ? $legacyPiece : ($unit * $wt); // سعر القطعة
    $sum   = $piece * $qty;
    $total += $sum;

    $items[] = [
        'id'     => $pid,
        'name'   => (string)$p['name'],
        'image'  => img_or_placeholder($p['image'] ?? ''),
        'metal'  => (string)($p['metal'] ?? 'gold'),
        'karat'  => $karat,
        'weight' => $wt,
        'unit'   => $unit,
        'piece'  => $piece,
        'qty'    => $qty,
        'sum'    => $sum,
    ];
}

// الحسابات البنكية
$accounts = $pdo->query("SELECT * FROM bank_accounts ORDER BY id ASC")->fetchAll(PDO::FETCH_ASSOC);

// عرض الصفحة
include __DIR__ . "/includes/header.php";
?>
<style>
.pay-wrap{padding:50px 0;width:90%;margin:0 auto}
.section-title{text-align:center;color:#eaeaea;font-size:28px;margin:0 0 18px;position:relative}
.section-title:after{content:'';display:block;width:60px;height:3px;background:var(--primary-color,#d4af37);margin:10px auto;border-radius:10px}
.note-empty{text-align:center;color:#cbd5e1;padding:40px 0}
.tbl{width:100%;border-collapse:separate;border-spacing:0 10px;margin-bottom:16px}
.tbl thead th{color:#d4af37;font-weight:800;text-align:right;padding:10px;border-bottom:1px solid rgba(212,175,55,.25)}
.tbl tbody tr{background:#121212;border:1px solid rgba(212,175,55,.25);border-radius:14px;box-shadow:0 8px 20px rgba(0,0,0,.35)}
.tbl tbody td{padding:10px;vertical-align:middle}
.tbl tbody tr > td:first-child{border-radius:14px 0 0 14px}
.tbl tbody tr > td:last-child{border-radius:0 14px 14px 0}
.thumb{width:80px;height:70px;object-fit:cover;border-radius:10px;border:1px solid rgba(212,175,55,.25);background:#000}
.chips{display:flex;gap:8px;flex-wrap:wrap}
.chip{padding:6px 10px;border-radius:999px;font-weight:800;font-size:.85rem;color:#0f0f0f;background:linear-gradient(145deg,#b68f2a,#d4af37,#c8a951)}
.chip.gray{background:#0b0b0b;color:#eaeaea;border:1px solid rgba(212,175,55,.35)}

.summary{margin:16px auto 8px;text-align:center;color:#eaeaea;font-size:20px;font-weight:900}
.pay-card{max-width:720px;margin:20px auto;background:linear-gradient(180deg,#141414 0%,#0f0f0f 100%);border:1px solid rgba(212,175,55,.35);border-radius:14px;box-shadow:0 12px 28px rgba(0,0,0,.35);padding:18px}
.pay-card h3{margin:0 0 12px;color:#eaeaea}
.field{margin-bottom:14px}
.field label{display:block;margin:0 0 6px;color:#ddd;font-weight:700}
.select,input[type="file"],textarea{
  width:100%;background:#0b0b0b;color:#eaeaea;border:1px solid rgba(212,175,55,.35);
  border-radius:10px;padding:10px;outline:0
}
textarea{min-height:90px;resize:vertical}
.btns{display:flex;gap:10px;flex-wrap:wrap;justify-content:center;margin-top:10px}
.btn{display:inline-flex;align-items:center;gap:8px;padding:12px 16px;border-radius:12px;font-weight:800;border:0;cursor:pointer;text-decoration:none}
.btn-primary{background:linear-gradient(145deg,#b68f2a,#d4af37,#c8a951);color:#0f0f0f}
.help{color:#9aa7b1;text-align:center;margin-top:8px;font-size:.95rem}
</style>

<section class="pay-wrap" dir="rtl">
  <h2 class="section-title">💳 الدفع عبر التحويل البنكي</h2>

  <!-- ملخص السلة -->
  <div style="overflow:auto">
    <table class="tbl" aria-label="ملخص السلة قبل الدفع">
      <thead>
        <tr>
          <th>#</th>
          <th>الصورة</th>
          <th>المنتج</th>
          <th>البيانات</th>
          <th>سعر القطعة</th>
          <th>الكمية</th>
          <th>المجموع</th>
        </tr>
      </thead>
      <tbody>
        <?php foreach ($items as $i => $it): ?>
          <tr>
            <td><?= $i+1 ?></td>
            <td><img class="thumb" src="<?= e($it['image']) ?>" alt=""></td>
            <td><?= e($it['name']) ?></td>
            <td>
              <div class="chips">
                <span class="chip"><?= $it['metal']==='gold' ? ('ذهب عيار '.$it['karat']) : 'فضة' ?></span>
                <span class="chip gray"><?= number_format($it['weight'],2) ?> جم</span>
                <?php if ($it['metal']==='gold'): ?>
                  <span class="chip gray"><?= number_format($prices['gold_'.$it['karat']] ?? 0, 2) ?> ر.س/جم</span>
                <?php else: ?>
                  <span class="chip gray"><?= number_format($prices['silver'] ?? 0, 2) ?> ر.س/جم</span>
                <?php endif; ?>
              </div>
            </td>
            <td><?= number_format($it['piece'],2) ?> ر.س</td>
            <td><?= (int)$it['qty'] ?></td>
            <td><strong><?= number_format($it['sum'],2) ?> ر.س</strong></td>
          </tr>
        <?php endforeach; ?>
      </tbody>
    </table>
  </div>

  <div class="summary">إجمالي الطلب: <?= number_format($total, 2) ?> ر.س</div>

  <!-- نموذج التحويل -->
  <form action="transfer_submit.php" method="post" enctype="multipart/form-data" class="pay-card">
    <h3>بيانات التحويل</h3>

    <div class="field">
      <label for="bank_account_id">اختر الحساب البنكي:</label>
      <select id="bank_account_id" name="bank_account_id" class="select" required>
        <option value="">-- اختر الحساب --</option>
        <?php foreach ($accounts as $acc): ?>
          <option value="<?= (int)$acc['id'] ?>">
            <?= e($acc['bank_name']) ?> — <?= e($acc['account_holder']) ?> (<?= e($acc['iban']) ?>)
          </option>
        <?php endforeach; ?>
      </select>
    </div>

    <div class="field">
      <label for="notes">ملاحظات (اختياري):</label>
      <textarea id="notes" name="notes" placeholder="اكتب أي ملاحظات تخص التحويل أو الاستلام..."></textarea>
    </div>

    <div class="field">
      <label for="receipt">رفع إيصال التحويل:</label>
      <input id="receipt" type="file" name="receipt" accept="image/*,application/pdf" required>
      <div class="help">الأنواع المسموحة: صور أو PDF. حجم الملف لا يفضل أن يتجاوز 5MB.</div>
    </div>

    <!-- حقول خفية مطلوبة للمعالجة -->
    <input type="hidden" name="csrf" value="<?= e($_SESSION['csrf']) ?>">
    <input type="hidden" name="total" value="<?= e($total) ?>">

    <div class="btns">
      <button type="submit" class="btn btn-primary"><i class="fa-solid fa-paper-plane"></i> إرسال الطلب</button>
    </div>

    <div class="help">بعد الإرسال، سنراجع التحويل ونتواصل معك لتأكيد الطلب.</div>
  </form>

  <?php if (!$accounts): ?>
    <div class="help" style="margin-top:10px">⚠️ لا توجد حسابات بنكية مُسجلة حاليًا. أضِفها من لوحة التحكم.</div>
  <?php endif; ?>
</section>

<?php include __DIR__ . "/includes/footer.php"; ?>
